<?php

namespace App\Http\Controllers;

use App\Business;
use App\BusinessLocation;
use App\Product;
use App\Transaction;
use App\Utils\ProductUtil;
use App\Variation;
use Carbon\Carbon;
use DB;
use Excel;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class ImportOpeningStockController extends Controller
{
    /**
     * All Utils instance.
     */
    protected $productUtil;

    /**
     * Constructor
     *
     * @param  ProductUtil  $productUtil
     * @return void
     */
    public function __construct(ProductUtil $productUtil)
    {
        $this->productUtil = $productUtil;
    }

    /**
     * Display import product screen.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        if (! auth()->user()->can('product.opening_stock')) {
            abort(403, 'Unauthorized action.');
        }

        $zip_loaded = extension_loaded('zip');

        $date_formats = Business::date_formats();
        $date_format = session('business.date_format');
        $date_format = isset($date_formats[$date_format]) ? $date_formats[$date_format] : $date_format;

        if ($zip_loaded === false) {
            $notification = [
                'success' => 0,
                'msg' => 'Please install/enable PHP Zip archive for import',
            ];

            return view('import_opening_stock.index')
                ->with(compact('notification', 'date_format'));
        } else {
            return view('import_opening_stock.index')
                ->with(compact('date_format'));
        }
    }

    /**
     * Imports the uploaded file to database.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        if (! auth()->user()->can('product.opening_stock')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $notAllowed = $this->productUtil->notAllowedInDemo();
            if (! empty($notAllowed)) {
                return $notAllowed;
            }

            //Set maximum php execution time
            ini_set('max_execution_time', 0);
            ini_set('memory_limit', -1);

            if ($request->hasFile('products_csv')) {
                $file = $request->file('products_csv');

                $parsed_array = Excel::toArray([], $file);
                if (empty($parsed_array) || empty($parsed_array[0])) {
                    throw new \Exception('Uploaded file appears to be empty.');
                }

                //Remove header row
                $imported_data = array_splice($parsed_array[0], 1);

                $business_id = $request->session()->get('user.business_id');
                $user_id = $request->session()->get('user.id');

                $is_valid = true;
                $error_msg = '';

                // pick up the business date format (e.g., "mm-dd-yyyy")
                $biz_fmt_setting = session('business.date_format', 'Y-m-d'); // may be tokens like mm-dd-yyyy

                DB::beginTransaction();

                foreach ($imported_data as $key => $value) {
                    $row_no = $key + 2; // +2 because array_splice removed header (header was row 1)

                    // Normalize columns to avoid "undefined offset" when trailing empties exist
                    $sku                     = isset($value[0]) ? $this->cleanString($value[0]) : '';
                    $location_name_raw       = isset($value[1]) ? $this->cleanString($value[1]) : '';
                    $quantity_raw            = isset($value[2]) ? $this->cleanString($value[2]) : '';
                    $unit_cost_raw           = isset($value[3]) ? $this->cleanString($value[3]) : '';
                    $lot_number              = isset($value[4]) ? $this->cleanString($value[4]) : null;
                    $expiry_date_raw         = isset($value[5]) ? $this->cleanString($value[5]) : null;

                    // 1) SKU check + fetch product/variation
                    if ($sku === '') {
                        $is_valid = false;
                        $error_msg = "PRODUCT SKU is required in row no. {$row_no}";
                        break;
                    }

                    $product_info = Variation::where('sub_sku', $sku)
                        ->join('products AS P', 'variations.product_id', '=', 'P.id')
                        ->leftjoin('tax_rates AS TR', 'P.tax', 'TR.id')
                        ->where('P.business_id', $business_id)
                        ->select([
                            'P.id',
                            'variations.id as variation_id',
                            'P.enable_stock',
                            'TR.amount as tax_percent',
                            'TR.id as tax_id',
                        ])
                        ->first();

                    if (empty($product_info)) {
                        $is_valid = false;
                        $error_msg = "Product with SKU '{$sku}' not found in row no. {$row_no}";
                        break;
                    } elseif ($product_info->enable_stock == 0) {
                        $is_valid = false;
                        $error_msg = "Manage Stock not enabled for product with SKU '{$sku}' in row no. {$row_no}";
                        break;
                    }

                    // 2) Location (default to first business location if blank)
                    if (! empty($location_name_raw)) {
                        $location = BusinessLocation::where('name', $location_name_raw)
                            ->where('business_id', $business_id)
                            ->first();

                        if (empty($location)) {
                            $is_valid = false;
                            $error_msg = "Location with name '{$location_name_raw}' not found in row no. {$row_no}";
                            break;
                        }
                    } else {
                        $location = BusinessLocation::where('business_id', $business_id)->first();
                    }

                    // 3) Quantity
                    $quantity = $this->parseNumber($quantity_raw);
                    if (! is_numeric($quantity)) {
                        $is_valid = false;
                        $error_msg = "Invalid quantity '{$quantity_raw}' in row no. {$row_no}";
                        break;
                    }

                    // 4) Unit cost
                    $unit_cost_before_tax = $this->parseNumber($unit_cost_raw);
                    if (! is_numeric($unit_cost_before_tax)) {
                        $is_valid = false;
                        $error_msg = "Invalid UNIT COST '{$unit_cost_raw}' in row no. {$row_no}";
                        break;
                    }

                    // 5) Expiry date (optional) — parse strictly by Business setting, tolerate -, /, time parts, BOM, etc.
                    $exp_date = null;
                    if (! empty($expiry_date_raw)) {
                        try {
                            $exp_date = $this->parseBusinessDate($expiry_date_raw, $biz_fmt_setting);
                        } catch (\Throwable $e) {
                            $is_valid = false;
                            $expect = $biz_fmt_setting;
                            $error_msg = "Invalid Expiry Date '{$expiry_date_raw}' in row no. {$row_no}. Expected {$expect}.";
                            break;
                        }
                    }

                    $opening_stock = [
                        'quantity'   => $quantity,
                        'location_id'=> $location->id,
                        'lot_number' => $lot_number ?: null,
                        'exp_date'   => $exp_date,
                    ];

                    // Check for existing opening stock transaction (same product & location)
                    $os_transaction = Transaction::where('business_id', $business_id)
                        ->where('location_id', $location->id)
                        ->where('type', 'opening_stock')
                        ->where('opening_stock_product_id', $product_info->id)
                        ->first();

                    $this->addOpeningStock(
                        $opening_stock,
                        $product_info,
                        $business_id,
                        (float)$unit_cost_before_tax,
                        $os_transaction
                    );
                }
            }

            if (! $is_valid) {
                throw new \Exception($error_msg);
            }

            DB::commit();

            $output = [
                'success' => 1,
                'msg' => __('product.file_imported_successfully'),
            ];

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency('File:'.$e->getFile().' Line:'.$e->getLine().' Message:'.$e->getMessage());

            $output = [
                'success' => 0,
                'msg' => 'Message:'.$e->getMessage(),
            ];

            return redirect('import-opening-stock')->with('notification', $output);
        }

        return redirect('import-opening-stock')->with('status', $output);
    }

    /**
     * Adds opening stock of a single product
     *
     * @param  array  $opening_stock
     * @param  object $product
     * @param  int    $business_id
     * @param  float  $unit_cost_before_tax
     * @param  \App\Transaction|null $transaction
     * @return void
     */
    private function addOpeningStock($opening_stock, $product, $business_id, $unit_cost_before_tax, $transaction = null)
    {
        $user_id = request()->session()->get('user.id');

        $transaction_date = request()->session()->get('financial_year.start');
        $transaction_date = Carbon::createFromFormat('Y-m-d', $transaction_date)->toDateTimeString();

        //Get product tax
        $tax_percent = ! empty($product->tax_percent) ? (float)$product->tax_percent : 0.0;
        $tax_id      = ! empty($product->tax_id) ? $product->tax_id : null;

        $item_tax = $this->productUtil->calc_percentage($unit_cost_before_tax, $tax_percent);

        //total before transaction tax
        $total_before_trans_tax = (float)$opening_stock['quantity'] * ((float)$unit_cost_before_tax + (float)$item_tax);

        //Add opening stock transaction
        if (empty($transaction)) {
            $transaction = new Transaction();
            $transaction->type = 'opening_stock';
            $transaction->status = 'received';
            $transaction->opening_stock_product_id = $product->id;
            $transaction->business_id = $business_id;
            $transaction->transaction_date = $transaction_date;
            $transaction->location_id = $opening_stock['location_id'];
            $transaction->payment_status = 'paid';
            $transaction->created_by = $user_id;
            $transaction->total_before_tax = 0;
            $transaction->final_total = 0;
        }

        $transaction->total_before_tax += $total_before_trans_tax;
        $transaction->final_total += $total_before_trans_tax;
        $transaction->save();

        //Create purchase line
        $transaction->purchase_lines()->create([
            'product_id'               => $product->id,
            'variation_id'             => $product->variation_id,
            'quantity'                 => $opening_stock['quantity'],
            'pp_without_discount'      => $unit_cost_before_tax,
            'item_tax'                 => $item_tax,
            'tax_id'                   => $tax_id,
            'purchase_price'           => $unit_cost_before_tax,
            'purchase_price_inc_tax'   => $unit_cost_before_tax + $item_tax,
            'exp_date'                 => ! empty($opening_stock['exp_date']) ? $opening_stock['exp_date'] : null,
            'lot_number'               => ! empty($opening_stock['lot_number']) ? $opening_stock['lot_number'] : null,
        ]);

        //Update variation location details
        $this->productUtil->updateProductQuantity(
            $opening_stock['location_id'],
            $product->id,
            $product->variation_id,
            $opening_stock['quantity']
        );
    }

    // ---------------------------
    // Helpers (single-file only)
    // ---------------------------

    /**
     * Clean string: remove BOM/zero-width, trim spaces.
     */
    private function cleanString($raw): string
    {
        if ($raw === null) return '';
        $val = preg_replace('/[^\PC\s]/u', '', (string)$raw); // strip BOM/zero-width
        $val = trim($val);
        // collapse internal excessive whitespace
        $val = preg_replace('/\s+/', ' ', $val);
        return $val;
    }

    /**
     * Parse a numeric text (removes commas, currency symbols, etc.)
     */
    private function parseNumber($raw)
    {
        if ($raw === null) return null;
        $s = $this->cleanString($raw);
        // remove currency and thousand separators
        $s = preg_replace('/[^\d\.\-]/', '', $s);
        // Handle numbers like "1,234.56" already cleaned above.
        if ($s === '' || $s === '-' || $s === '.') return null;
        return is_numeric($s) ? 0 + $s : null;
    }

    /**
     * Convert Business date setting (e.g., "mm-dd-yyyy") to Carbon format (e.g., "m-d-Y")
     */
    private function toCarbonFmt(string $bizSetting): string
    {
        $s = strtolower(trim($bizSetting));     // e.g., "mm-dd-yyyy"
        // replace longest tokens first to avoid overlaps
        $s = str_replace(['yyyy','yy','dd','mm'], ['Y','y','d','m'], $s);
        return $s; // keep separators (- / .) intact
    }

    /**
     * Parse user-supplied date using Business date format; normalize to "Y-m-d".
     * Tolerates -, /, . separators and optional time parts.
     *
     * @throws \InvalidArgumentException if parsing fails
     */
    private function parseBusinessDate(string $raw, ?string $bizSetting): string
    {
        $val = $this->cleanString($raw);
        if ($val === '' || Str::lower($val) === 'n/a') {
            throw new \InvalidArgumentException("Empty date");
        }

        $baseFmt = $this->toCarbonFmt($bizSetting ?? 'Y-m-d'); // e.g., "m-d-Y"

        // 1) exact format
        try {
            return Carbon::createFromFormat($baseFmt, $val)->format('Y-m-d');
        } catch (\Throwable $e) {}

        // 2) accept common time suffixes
        foreach ([' H:i:s',' H:i'] as $t) {
            try {
                return Carbon::createFromFormat($baseFmt.$t, $val)->format('Y-m-d');
            } catch (\Throwable $e) {}
        }

        // 3) tolerate separator variants both in format and value
        foreach (['-','/','.'] as $sep) {
            $fmt = preg_replace('/[-\/.]/', $sep, $baseFmt);
            $v   = preg_replace('/[-\/.]/', $sep, $val);
            try {
                return Carbon::createFromFormat($fmt, $v)->format('Y-m-d');
            } catch (\Throwable $e) {}
            // with time parts
            foreach ([' H:i:s',' H:i'] as $t) {
                try {
                    return Carbon::createFromFormat($fmt.$t, $v)->format('Y-m-d');
                } catch (\Throwable $e) {}
            }
        }

        // 4) last resort smart parser (ISO-like)
        try {
            return Carbon::parse($val)->format('Y-m-d');
        } catch (\Throwable $e) {}

        throw new \InvalidArgumentException("Invalid date '{$raw}'");
    }
}
